<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\TranslationHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Project\ProjectRequest;
use App\Models\Project;
use App\Models\Category;
use App\Models\Tag;
use App\Services\ProjectService;
use App\ViewModels\ProjectViewModel;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Yajra\DataTables\DataTables;

class ProjectController extends Controller
{
    private $project;

    public function __construct()
    {
        $this->middleware(['auth:admin', 'permission:projects.view_all'], ['only' => ['index']]);
        $this->middleware(['auth:admin', 'permission:projects.view_details'], ['only' => ['show']]);
        $this->middleware(['auth:admin', 'permission:projects.create'], ['only' => ['create', 'update']]);
        $this->middleware(['auth:admin', 'permission:projects.edit'], ['only' => ['edit', 'store']]);
        $this->middleware(['auth:admin', 'permission:projects.delete'], ['only' => ['destroy']]);
        $this->middleware(['auth:admin', 'permission:projects.sort'], ['only' => ['reorder', 'saveReorder']]);
        $this->project = new ProjectService();
    }
    
    /**
     * Display a listing of the project.
     */
    public function index(Request $request): View
    {
        return view('admin.pages.project.index');
    }


    public function table(DataTables $dataTables, Request $request)
    {
        $model = Project::ordered();

        return $dataTables->eloquent($model)->addIndexColumn()
            ->filter(function ($query) use ($request) {
                if ($search = $request->get('search')['value']) { // البحث باستخدام الاسم
                    $query->where(function ($q) use ($search) {
                        $q->where('name', 'LIKE', "%$search%");
                    });
                }
            })
            ->editColumn('id', function (Project $project) {
                return $project->id ?? '-';
            })
            ->editColumn('name', function (Project $project) {
                return $project->name;
            })
            ->editColumn('category', function (Project $project) {
                return $project->categories->name ?? '-'; // عرض اسم القسم
            })
            ->editColumn('image', function (Project $project) {
                $imageUrl = $project->getFirstMediaUrl('projects_image') ?: asset('path/to/default/image.jpg');
                return "<img width='100' src='{$imageUrl}' alt='Project Image'/>";
            })
            ->editColumn('created_at', function (Project $project) {
                return $project->created_at ? $project->created_at->format('d-m-Y h:i A') : '-';
            })
            ->addColumn('is_active', function (Project $project) {
                return $project->is_active
                    ? '<label class="switch"><input type="checkbox" class="active-toggle" data-id="' . $project->id . '" checked><span class="slider round"></span></label>'
                    : '<label class="switch"><input type="checkbox" class="active-toggle" data-id="' . $project->id . '"><span class="slider round"></span></label>';
            })
            ->addColumn('action', function (Project $project) {
                return view('admin.pages.project.buttons', compact('project'));
            })
            ->rawColumns(['image', 'action', 'is_active'])
            ->startsWithSearch()
            ->make(true);
    }

    /**
     * Display a Single Row of the resource.
     */
    public function Show($id): View
    {

        $project = Project::find($id);
        return view('admin.pages.project.show', get_defined_vars());
    }

    public function toggleActiveStatus(Request $request)
    {
    $project = Project::findOrFail($request->id);
    $project->is_active = !$project->is_active;
    $project->save();
        return response()->json(['success' => true, 'is_active' => $project->is_active]);
    }


    /**
     * Show the form for creating a new project.
     */
    public function create(): View
    {
        $tags = Tag::get();
        return view('admin.pages.project.form',  new ProjectViewModel(), get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProjectRequest $request)
    {
        // return $request;
        $this->project->createProject($request->validated());
        session()->flash('success', TranslationHelper::translate('Operation Success'));
        return back();
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Project $project): View
    {
       
        return view('admin.pages.project.form',  new ProjectViewModel($project), get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProjectRequest $request, Project $project): RedirectResponse
    {
        //    dd( $request->validated());
        $this->project->updateProject($project, $request->validated());
        session()->flash('success', TranslationHelper::translate('Successfully Updated'));
        return back();
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Project $project): JsonResponse
    {
        $this->project->deleteProject($project);
        return response()->json(['status' => 'success', 'message' => TranslationHelper::translate('Successfully Deleted')]);
    }


    public function reorder(Project $project)
    {
        return $this->project->reorder($project, 'name', 'admin.pages.project.reorder', 1);
    }

    public function saveReorder(Request $request, Project $project)
    {
        $all_entries = $request->input('tree');
        return $this->project->saveReorder($all_entries, $project);
    }

}
